<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/CSVImporter.php';

session_start();

if (!isset($_SESSION['admin_authenticated']) || !$_SESSION['admin_authenticated']) {
    header('Location: /odds-predictor/admin/index.php');
    exit;
}

$conn = getDBConnection();
$message = '';
$error = '';

// Handle CSV Upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['csv_file'])) {
    $league_id = isset($_POST['league_id']) ? (int)$_POST['league_id'] : 0;
    $season = isset($_POST['season']) ? sanitizeInput($_POST['season']) : '';
    
    if (!$league_id || !$season) {
        $error = 'Please select league and season';
    } elseif ($_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
        $error = 'File upload error';
    } else {
        $file_path = $_FILES['csv_file']['tmp_name'];
        $importer = new CSVImporter($conn);
        $result = $importer->importCSV($file_path, $league_id, $season);
        
        if ($result['success']) {
            $message = "Import completed! Processed: {$result['rows_processed']}, Successful: {$result['rows_successful']}, Failed: {$result['rows_failed']}";
        } else {
            $error = $result['error'];
        }
    }
}

// Handle Ad Configuration
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_ads') {
    $custom_ad_enabled = isset($_POST['custom_ad_enabled']) ? 1 : 0;
    $network_ad_enabled = isset($_POST['network_ad_enabled']) ? 1 : 0;
    
    // Update or insert custom ad
    if ($custom_ad_enabled && !empty($_POST['custom_ad_code'])) {
        $ad_code = $_POST['custom_ad_code'];
        $url = $_POST['custom_ad_url'] ?? '';
        $duration = (int)($_POST['custom_ad_duration'] ?? 5);
        
        $query = "INSERT INTO ad_config (ad_type, ad_name, ad_code, position, enabled, url, duration_seconds) 
                  VALUES ('custom', 'Custom Ad', ?, 'top', ?, ?, ?) 
                  ON DUPLICATE KEY UPDATE ad_code=VALUES(ad_code), enabled=VALUES(enabled), url=VALUES(url), duration_seconds=VALUES(duration_seconds)";
        $stmt = $conn->prepare($query);
        $stmt->bind_param("sisi", $ad_code, $custom_ad_enabled, $url, $duration);
        $stmt->execute();
    }
    
    // Update network ad
    if ($network_ad_enabled && !empty($_POST['network_ad_code'])) {
        $ad_code = $_POST['network_ad_code'];
        
        $query = "INSERT INTO ad_config (ad_type, ad_name, ad_code, position, enabled) 
                  VALUES ('network', 'Network Ad', ?, 'middle', ?) 
                  ON DUPLICATE KEY UPDATE ad_code=VALUES(ad_code), enabled=VALUES(enabled)";
        $stmt = $conn->prepare($query);
        $stmt->bind_param("si", $ad_code, $network_ad_enabled);
        $stmt->execute();
    }
    
    $message = 'Ad configuration updated';
}

// Get leagues
$leagues_query = "SELECT * FROM leagues ORDER BY country, name";
$leagues_result = $conn->query($leagues_query);
$leagues = [];
while ($row = $leagues_result->fetch_assoc()) {
    $leagues[] = $row;
}

// Get upload history
$uploads_query = "SELECT * FROM csv_uploads ORDER BY upload_date DESC LIMIT 10";
$uploads_result = $conn->query($uploads_query);
$uploads = [];
while ($row = $uploads_result->fetch_assoc()) {
    $uploads[] = $row;
}

// Get data config
$config_query = "SELECT * FROM data_config";
$config_result = $conn->query($config_query);
$config = [];
while ($row = $config_result->fetch_assoc()) {
    $config[$row['config_key']] = $row['config_value'];
}

// Get ad config
$ads_query = "SELECT * FROM ad_config";
$ads_result = $conn->query($ads_query);
$ads = [];
while ($row = $ads_result->fetch_assoc()) {
    $ads[$row['ad_type']] = $row;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="/odds-predictor/css/style.css">
    <style>
        .admin-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .admin-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            background: white;
            padding: 20px;
            border-radius: 10px;
        }
        
        .admin-header h1 {
            color: var(--primary-color);
        }
        
        .admin-header a {
            background: var(--secondary-color);
            color: white;
            padding: 10px 20px;
            border-radius: 5px;
            text-decoration: none;
        }
        
        .admin-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            border-bottom: 2px solid var(--border-color);
        }
        
        .admin-tabs button {
            background: none;
            border: none;
            padding: 15px 20px;
            cursor: pointer;
            font-size: 16px;
            color: var(--text-light);
            border-bottom: 3px solid transparent;
            transition: all 0.3s;
        }
        
        .admin-tabs button.active {
            color: var(--primary-color);
            border-bottom-color: var(--primary-color);
        }
        
        .tab-content {
            display: none;
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }
        
        .tab-content.active {
            display: block;
        }
        
        .message {
            background: #e8f5e9;
            color: #2e7d32;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        
        .error {
            background: #ffebee;
            color: #c62828;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        
        .upload-form {
            background: var(--light-bg);
            padding: 20px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        
        .upload-form .form-group {
            margin-bottom: 15px;
        }
        
        .upload-form label {
            display: block;
            margin-bottom: 5px;
            font-weight: 600;
        }
        
        .upload-form input,
        .upload-form select {
            width: 100%;
            padding: 10px;
            border: 1px solid var(--border-color);
            border-radius: 5px;
        }
        
        .upload-form button {
            background: var(--primary-color);
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-weight: 600;
        }
        
        .upload-form button:hover {
            background: #1565c0;
        }
        
        .upload-history {
            margin-top: 30px;
        }
        
        .upload-history table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .upload-history th,
        .upload-history td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid var(--border-color);
        }
        
        .upload-history th {
            background: var(--light-bg);
            font-weight: 600;
            color: var(--primary-color);
        }
        
        .status-badge {
            padding: 5px 10px;
            border-radius: 3px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .status-completed {
            background: #e8f5e9;
            color: #2e7d32;
        }
        
        .status-processing {
            background: #fff3e0;
            color: #e65100;
        }
        
        .status-error {
            background: #ffebee;
            color: #c62828;
        }
        
        .data-config {
            background: var(--light-bg);
            padding: 20px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        
        .data-config p {
            margin: 10px 0;
        }
        
        .ad-config-section {
            background: var(--light-bg);
            padding: 20px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        
        .ad-config-section h3 {
            color: var(--primary-color);
            margin-bottom: 15px;
        }
        
        .ad-config-section textarea {
            width: 100%;
            min-height: 150px;
            padding: 10px;
            border: 1px solid var(--border-color);
            border-radius: 5px;
            font-family: monospace;
            font-size: 12px;
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <div class="admin-header">
            <h1>Admin Dashboard</h1>
            <a href="/odds-predictor/public/index.php">← Back to App</a>
        </div>
        
        <?php if ($message): ?>
        <div class="message"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
        <div class="error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <div class="admin-tabs">
            <button class="tab-btn active" onclick="switchTab('import')">CSV Import</button>
            <button class="tab-btn" onclick="switchTab('config')">Data Config</button>
            <button class="tab-btn" onclick="switchTab('ads')">Ad Management</button>
        </div>
        
        <!-- CSV Import Tab -->
        <div id="import" class="tab-content active">
            <h2>Import Match Data</h2>
            
            <div class="upload-form">
                <form method="POST" enctype="multipart/form-data">
                    <div class="form-group">
                        <label for="league_id">Select League:</label>
                        <select name="league_id" required>
                            <option value="">-- Select League --</option>
                            <?php foreach ($leagues as $league): ?>
                            <option value="<?php echo $league['id']; ?>">
                                <?php echo htmlspecialchars($league['name']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="season">Season (e.g., 2024/25):</label>
                        <input type="text" name="season" placeholder="2024/25" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="csv_file">CSV File:</label>
                        <input type="file" name="csv_file" accept=".csv" required>
                        <p style="font-size: 12px; color: var(--text-light); margin-top: 5px;">
                            <a href="/odds-predictor/data/CSV_TEMPLATE.md" target="_blank">📋 View CSV Template</a>
                        </p>
                    </div>
                    
                    <button type="submit">Upload & Import</button>
                </form>
            </div>
            
            <div class="upload-history">
                <h3>Recent Uploads</h3>
                <table>
                    <thead>
                        <tr>
                            <th>Filename</th>
                            <th>Date</th>
                            <th>Processed</th>
                            <th>Successful</th>
                            <th>Failed</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($uploads as $upload): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($upload['filename']); ?></td>
                            <td><?php echo date('Y-m-d H:i', strtotime($upload['upload_date'])); ?></td>
                            <td><?php echo $upload['rows_processed']; ?></td>
                            <td><?php echo $upload['rows_successful']; ?></td>
                            <td><?php echo $upload['rows_failed']; ?></td>
                            <td>
                                <span class="status-badge status-<?php echo $upload['status']; ?>">
                                    <?php echo ucfirst($upload['status']); ?>
                                </span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- Data Config Tab -->
        <div id="config" class="tab-content">
            <h2>Data Configuration</h2>
            
            <div class="data-config">
                <h3>Current Data Range</h3>
                <p><strong>Minimum Season:</strong> <?php echo htmlspecialchars($config['min_season'] ?? 'N/A'); ?></p>
                <p><strong>Maximum Season:</strong> <?php echo htmlspecialchars($config['max_season'] ?? 'N/A'); ?></p>
                <p><strong>Recommended Range:</strong> <?php echo htmlspecialchars($config['recommended_range'] ?? 'N/A'); ?></p>
                <p><strong>Last Update:</strong> <?php echo htmlspecialchars($config['last_data_update'] ?? 'N/A'); ?></p>
                <p><strong>Freshness Warning Days:</strong> <?php echo htmlspecialchars($config['data_freshness_warning_days'] ?? 30); ?></p>
            </div>
        </div>
        
        <!-- Ad Management Tab -->
        <div id="ads" class="tab-content">
            <h2>Ad Management</h2>
            
            <form method="POST">
                <input type="hidden" name="action" value="update_ads">
                
                <div class="ad-config-section">
                    <h3>Custom Ad (Your Own)</h3>
                    <div class="form-group">
                        <label>
                            <input type="checkbox" name="custom_ad_enabled" <?php echo (isset($ads['custom']) && $ads['custom']['enabled']) ? 'checked' : ''; ?>>
                            Enable Custom Ad
                        </label>
                    </div>
                    <div class="form-group">
                        <label for="custom_ad_code">Ad Code (HTML/JavaScript):</label>
                        <textarea name="custom_ad_code"><?php echo isset($ads['custom']) ? htmlspecialchars($ads['custom']['ad_code']) : ''; ?></textarea>
                    </div>
                    <div class="form-group">
                        <label for="custom_ad_url">Ad URL (optional):</label>
                        <input type="url" name="custom_ad_url" value="<?php echo isset($ads['custom']) ? htmlspecialchars($ads['custom']['url']) : ''; ?>">
                    </div>
                    <div class="form-group">
                        <label for="custom_ad_duration">Duration (seconds):</label>
                        <input type="number" name="custom_ad_duration" value="<?php echo isset($ads['custom']) ? $ads['custom']['duration_seconds'] : '5'; ?>" min="1">
                    </div>
                </div>
                
                <div class="ad-config-section">
                    <h3>Network Ad (PropellerAds, Adsterra, etc.)</h3>
                    <div class="form-group">
                        <label>
                            <input type="checkbox" name="network_ad_enabled" <?php echo (isset($ads['network']) && $ads['network']['enabled']) ? 'checked' : ''; ?>>
                            Enable Network Ad
                        </label>
                    </div>
                    <div class="form-group">
                        <label for="network_ad_code">Ad Code (HTML/JavaScript):</label>
                        <textarea name="network_ad_code"><?php echo isset($ads['network']) ? htmlspecialchars($ads['network']['ad_code']) : ''; ?></textarea>
                    </div>
                </div>
                
                <button type="submit" class="btn btn-primary">Update Ad Configuration</button>
            </form>
        </div>
    </div>
    
    <script>
        function switchTab(tabName) {
            // Hide all tabs
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Remove active class from all buttons
            document.querySelectorAll('.tab-btn').forEach(btn => {
                btn.classList.remove('active');
            });
            
            // Show selected tab
            document.getElementById(tabName).classList.add('active');
            
            // Add active class to clicked button
            event.target.classList.add('active');
        }
    </script>
</body>
</html>
