<?php
/**
 * CSV Importer - Handles bulk data uploads
 * Supports football-data.co.uk format
 */

class CSVImporter {
    private $conn;
    private $upload_id;
    private $rows_processed = 0;
    private $rows_successful = 0;
    private $rows_failed = 0;
    private $errors = [];
    
    public function __construct($conn) {
        $this->conn = $conn;
    }
    
    /**
     * Import CSV file
     * Expected format: Date, HomeTeam, AwayTeam, FTHG, FTAG, FTR, HS, AS, HST, AST, HF, AF, HC, AC, B365H, B365D, B365A, etc.
     */
    public function importCSV($file_path, $league_id, $season) {
        if (!file_exists($file_path)) {
            return ['success' => false, 'error' => 'File not found'];
        }
        
        // Log upload
        $this->upload_id = $this->logUploadStart(basename($file_path));
        
        $file = fopen($file_path, 'r');
        if (!$file) {
            return ['success' => false, 'error' => 'Cannot open file'];
        }
        
        $header = fgetcsv($file);
        $column_map = $this->mapColumns($header);
        
        while (($row = fgetcsv($file)) !== false) {
            $this->rows_processed++;
            
            try {
                $this->processRow($row, $column_map, $league_id, $season);
                $this->rows_successful++;
            } catch (Exception $e) {
                $this->rows_failed++;
                $this->errors[] = "Row {$this->rows_processed}: " . $e->getMessage();
            }
        }
        
        fclose($file);
        
        // Update upload log
        $this->logUploadComplete();
        
        return [
            'success' => true,
            'rows_processed' => $this->rows_processed,
            'rows_successful' => $this->rows_successful,
            'rows_failed' => $this->rows_failed,
            'errors' => $this->errors
        ];
    }
    
    /**
     * Map CSV columns to database fields
     */
    private function mapColumns($header) {
        $map = [];
        $expected_columns = [
            'Date', 'HomeTeam', 'AwayTeam', 'FTHG', 'FTAG', 'FTR',
            'HS', 'AS', 'HST', 'AST', 'HF', 'AF', 'HC', 'AC',
            'B365H', 'B365D', 'B365A'
        ];
        
        foreach ($expected_columns as $col) {
            $key = array_search($col, $header);
            if ($key !== false) {
                $map[$col] = $key;
            }
        }
        
        return $map;
    }
    
    /**
     * Process individual row
     */
    private function processRow($row, $column_map, $league_id, $season) {
        // Extract data
        $date = isset($column_map['Date']) ? $row[$column_map['Date']] : null;
        $home_team = isset($column_map['HomeTeam']) ? $row[$column_map['HomeTeam']] : null;
        $away_team = isset($column_map['AwayTeam']) ? $row[$column_map['AwayTeam']] : null;
        $home_goals = isset($column_map['FTHG']) ? (int)$row[$column_map['FTHG']] : null;
        $away_goals = isset($column_map['FTAG']) ? (int)$row[$column_map['FTAG']] : null;
        $result = isset($column_map['FTR']) ? $row[$column_map['FTR']] : null;
        
        // Validate required fields
        if (!$date || !$home_team || !$away_team) {
            throw new Exception("Missing required fields");
        }
        
        // Get or create teams
        $home_team_id = $this->getOrCreateTeam($home_team, $league_id);
        $away_team_id = $this->getOrCreateTeam($away_team, $league_id);
        
        // Parse date
        $match_date = $this->parseDate($date);
        
        // Extract statistics
        $home_shots = isset($column_map['HS']) ? (int)$row[$column_map['HS']] : null;
        $away_shots = isset($column_map['AS']) ? (int)$row[$column_map['AS']] : null;
        $home_shots_on_target = isset($column_map['HST']) ? (int)$row[$column_map['HST']] : null;
        $away_shots_on_target = isset($column_map['AST']) ? (int)$row[$column_map['AST']] : null;
        
        // Extract odds
        $odds_home = isset($column_map['B365H']) ? (float)$row[$column_map['B365H']] : null;
        $odds_draw = isset($column_map['B365D']) ? (float)$row[$column_map['B365D']] : null;
        $odds_away = isset($column_map['B365A']) ? (float)$row[$column_map['B365A']] : null;
        
        // Insert match
        $query = "INSERT INTO matches (league_id, home_team_id, away_team_id, match_date, season, home_goals, away_goals, result, home_shots, away_shots, home_shots_on_target, away_shots_on_target, odds_home, odds_draw, odds_away, bookmaker) 
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Bet365')
                  ON DUPLICATE KEY UPDATE home_goals=VALUES(home_goals), away_goals=VALUES(away_goals)";
        
        $stmt = $this->conn->prepare($query);
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $this->conn->error);
        }
        
        $stmt->bind_param("iiissiisiiiddd", 
            $league_id, $home_team_id, $away_team_id, $match_date, $season,
            $home_goals, $away_goals, $result,
            $home_shots, $away_shots, $home_shots_on_target, $away_shots_on_target,
            $odds_home, $odds_draw, $odds_away
        );
        
        if (!$stmt->execute()) {
            throw new Exception("Execute failed: " . $stmt->error);
        }
        
        // Update team statistics
        $this->updateTeamStats($home_team_id, $league_id, $season, $home_goals, $away_goals, 'home', $home_shots, $home_shots_on_target);
        $this->updateTeamStats($away_team_id, $league_id, $season, $away_goals, $home_goals, 'away', $away_shots, $away_shots_on_target);
        
        // Update H2H record
        $this->updateH2HRecord($home_team_id, $away_team_id, $result);
    }
    
    /**
     * Get or create team
     */
    private function getOrCreateTeam($team_name, $league_id) {
        // First try to find existing team
        $query = "SELECT id FROM teams WHERE name = ? AND league_id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("si", $team_name, $league_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($row = $result->fetch_assoc()) {
            return $row['id'];
        }
        
        // Create new team
        $query = "INSERT INTO teams (name, league_id) VALUES (?, ?)";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("si", $team_name, $league_id);
        $stmt->execute();
        
        return $this->conn->insert_id;
    }
    
    /**
     * Parse date from various formats
     */
    private function parseDate($date_str) {
        $formats = ['d/m/Y', 'Y-m-d', 'd-m-Y', 'm/d/Y'];
        
        foreach ($formats as $format) {
            $parsed = DateTime::createFromFormat($format, $date_str);
            if ($parsed) {
                return $parsed->format('Y-m-d');
            }
        }
        
        throw new Exception("Invalid date format: $date_str");
    }
    
    /**
     * Update team statistics
     */
    private function updateTeamStats($team_id, $league_id, $season, $goals_for, $goals_against, $position, $shots, $shots_on_target) {
        $wins = 0;
        $draws = 0;
        $losses = 0;
        
        if ($goals_for > $goals_against) {
            $wins = 1;
        } elseif ($goals_for == $goals_against) {
            $draws = 1;
        } else {
            $losses = 1;
        }
        
        $query = "INSERT INTO team_stats (team_id, season, league_id, matches_played, wins, draws, losses, goals_for, goals_against, shots_for, shots_on_target_for)
                  VALUES (?, ?, ?, 1, ?, ?, ?, ?, ?, ?, ?)
                  ON DUPLICATE KEY UPDATE 
                  matches_played = matches_played + 1,
                  wins = wins + VALUES(wins),
                  draws = draws + VALUES(draws),
                  losses = losses + VALUES(losses),
                  goals_for = goals_for + VALUES(goals_for),
                  goals_against = goals_against + VALUES(goals_against),
                  shots_for = shots_for + VALUES(shots_for),
                  shots_on_target_for = shots_on_target_for + VALUES(shots_on_target_for)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("isisiiiii", $team_id, $season, $league_id, $wins, $draws, $losses, $goals_for, $goals_against, $shots, $shots_on_target);
        $stmt->execute();
    }
    
    /**
     * Update head-to-head record
     */
    private function updateH2HRecord($team1_id, $team2_id, $result) {
        $team1_wins = 0;
        $team2_wins = 0;
        $draws = 0;
        
        if ($result == 'H') {
            $team1_wins = 1;
        } elseif ($result == 'A') {
            $team2_wins = 1;
        } else {
            $draws = 1;
        }
        
        $query = "INSERT INTO h2h_records (team1_id, team2_id, team1_wins, team2_wins, draws)
                  VALUES (?, ?, ?, ?, ?)
                  ON DUPLICATE KEY UPDATE 
                  team1_wins = team1_wins + VALUES(team1_wins),
                  team2_wins = team2_wins + VALUES(team2_wins),
                  draws = draws + VALUES(draws)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("iiiii", $team1_id, $team2_id, $team1_wins, $team2_wins, $draws);
        $stmt->execute();
    }
    
    /**
     * Log upload start
     */
    private function logUploadStart($filename) {
        $query = "INSERT INTO csv_uploads (filename, status) VALUES (?, 'processing')";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("s", $filename);
        $stmt->execute();
        return $this->conn->insert_id;
    }
    
    /**
     * Log upload completion
     */
    private function logUploadComplete() {
        $status = $this->rows_failed == 0 ? 'completed' : 'completed_with_errors';
        $error_log = implode("\n", $this->errors);
        
        $query = "UPDATE csv_uploads SET status = ?, rows_processed = ?, rows_successful = ?, rows_failed = ?, error_log = ? WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("siiiisi", $status, $this->rows_processed, $this->rows_successful, $this->rows_failed, $error_log, $this->upload_id);
        $stmt->execute();
    }
}
?>
