<?php
/**
 * Prediction Engine - Multiple Statistical Models
 * Implements: Poisson, Win %, xG, and Form Analysis
 */

class PredictionEngine {
    private $conn;
    private $home_team_id;
    private $away_team_id;
    private $season;
    
    public function __construct($conn, $home_team_id, $away_team_id, $season) {
        $this->conn = $conn;
        $this->home_team_id = $home_team_id;
        $this->away_team_id = $away_team_id;
        $this->season = $season;
    }
    
    /**
     * Model 1: Poisson Distribution
     * Uses historical goals scored and conceded to predict match outcomes
     */
    public function poissonPrediction() {
        $home_stats = $this->getTeamStats($this->home_team_id);
        $away_stats = $this->getTeamStats($this->away_team_id);
        
        if (!$home_stats || !$away_stats) {
            return ['error' => 'Insufficient data'];
        }
        
        // Calculate average goals per match
        $home_avg_goals = $home_stats['goals_for'] / max($home_stats['matches_played'], 1);
        $away_avg_goals = $away_stats['goals_for'] / max($away_stats['matches_played'], 1);
        $home_avg_conceded = $home_stats['goals_against'] / max($home_stats['matches_played'], 1);
        $away_avg_conceded = $away_stats['goals_against'] / max($away_stats['matches_played'], 1);
        
        // Adjust for home advantage (typically 0.3-0.4 goals)
        $home_lambda = ($home_avg_goals + $away_avg_conceded) / 2 + 0.35;
        $away_lambda = ($away_avg_goals + $home_avg_conceded) / 2;
        
        // Calculate probabilities using Poisson distribution
        $home_win_prob = 0;
        $draw_prob = 0;
        $away_win_prob = 0;
        
        for ($h = 0; $h <= 5; $h++) {
            for ($a = 0; $a <= 5; $a++) {
                $prob = $this->poissonProbability($h, $home_lambda) * $this->poissonProbability($a, $away_lambda);
                
                if ($h > $a) {
                    $home_win_prob += $prob;
                } elseif ($h == $a) {
                    $draw_prob += $prob;
                } else {
                    $away_win_prob += $prob;
                }
            }
        }
        
        // Normalize probabilities
        $total = $home_win_prob + $draw_prob + $away_win_prob;
        $home_win_prob = ($home_win_prob / $total) * 100;
        $draw_prob = ($draw_prob / $total) * 100;
        $away_win_prob = ($away_win_prob / $total) * 100;
        
        $prediction = $this->getPredictionFromProbabilities($home_win_prob, $draw_prob, $away_win_prob);
        
        return [
            'model' => 'Poisson Distribution',
            'prediction' => $prediction,
            'home_win_prob' => round($home_win_prob, 2),
            'draw_prob' => round($draw_prob, 2),
            'away_win_prob' => round($away_win_prob, 2),
            'confidence' => round(max($home_win_prob, $draw_prob, $away_win_prob), 2)
        ];
    }
    
    /**
     * Model 2: Win Percentage Analysis
     * Based on historical win rates and head-to-head records
     */
    public function winPercentagePrediction() {
        $home_stats = $this->getTeamStats($this->home_team_id);
        $away_stats = $this->getTeamStats($this->away_team_id);
        
        if (!$home_stats || !$away_stats) {
            return ['error' => 'Insufficient data'];
        }
        
        // Calculate win percentages
        $home_win_pct = ($home_stats['wins'] / max($home_stats['matches_played'], 1)) * 100;
        $away_win_pct = ($away_stats['wins'] / max($away_stats['matches_played'], 1)) * 100;
        
        // Get head-to-head data
        $h2h = $this->getH2HRecord();
        
        // Combine statistics
        $home_strength = ($home_win_pct * 0.6) + (($h2h['team1_wins'] / max($h2h['total_h2h'], 1)) * 100 * 0.4);
        $away_strength = ($away_win_pct * 0.6) + (($h2h['team2_wins'] / max($h2h['total_h2h'], 1)) * 100 * 0.4);
        
        // Add home advantage
        $home_strength += 5;
        
        $total_strength = $home_strength + $away_strength;
        $home_prob = ($home_strength / $total_strength) * 100;
        $away_prob = ($away_strength / $total_strength) * 100;
        $draw_prob = 100 - $home_prob - $away_prob;
        
        $prediction = $this->getPredictionFromProbabilities($home_prob, $draw_prob, $away_prob);
        
        return [
            'model' => 'Win Percentage Analysis',
            'prediction' => $prediction,
            'home_win_prob' => round($home_prob, 2),
            'draw_prob' => round($draw_prob, 2),
            'away_win_prob' => round($away_prob, 2),
            'confidence' => round(max($home_prob, $draw_prob, $away_prob), 2)
        ];
    }
    
    /**
     * Model 3: Expected Goals (xG) Analysis
     * Based on shot quality and quantity
     */
    public function xgPrediction() {
        $home_stats = $this->getTeamStats($this->home_team_id);
        $away_stats = $this->getTeamStats($this->away_team_id);
        
        if (!$home_stats || !$away_stats) {
            return ['error' => 'Insufficient data'];
        }
        
        // Use xG data if available, otherwise estimate from shots
        $home_xg = $home_stats['xg_for'] > 0 ? $home_stats['xg_for'] / max($home_stats['matches_played'], 1) : 
                   ($home_stats['shots_on_target_for'] * 0.08);
        $away_xg = $away_stats['xg_for'] > 0 ? $away_stats['xg_for'] / max($away_stats['matches_played'], 1) : 
                   ($away_stats['shots_on_target_for'] * 0.08);
        
        $home_xg_against = $home_stats['xg_against'] > 0 ? $home_stats['xg_against'] / max($home_stats['matches_played'], 1) : 
                           ($home_stats['shots_on_target_against'] * 0.08);
        $away_xg_against = $away_stats['xg_against'] > 0 ? $away_stats['xg_against'] / max($away_stats['matches_played'], 1) : 
                           ($away_stats['shots_on_target_against'] * 0.08);
        
        // Expected goals for this match
        $home_expected = ($home_xg + $away_xg_against) / 2 + 0.2; // Home advantage
        $away_expected = ($away_xg + $home_xg_against) / 2;
        
        // Convert xG to win probability
        $home_win_prob = $this->xgToWinProbability($home_expected, $away_expected);
        $away_win_prob = $this->xgToWinProbability($away_expected, $home_expected);
        $draw_prob = 100 - $home_win_prob - $away_win_prob;
        
        $prediction = $this->getPredictionFromProbabilities($home_win_prob, $draw_prob, $away_win_prob);
        
        return [
            'model' => 'Expected Goals (xG)',
            'prediction' => $prediction,
            'home_win_prob' => round($home_win_prob, 2),
            'draw_prob' => round($draw_prob, 2),
            'away_win_prob' => round($away_win_prob, 2),
            'home_xg' => round($home_expected, 2),
            'away_xg' => round($away_expected, 2),
            'confidence' => round(max($home_win_prob, $draw_prob, $away_win_prob), 2)
        ];
    }
    
    /**
     * Model 4: Recent Form Analysis
     * Based on last 5-10 matches performance
     */
    public function formPrediction() {
        $home_form = $this->getRecentForm($this->home_team_id, 10);
        $away_form = $this->getRecentForm($this->away_team_id, 10);
        
        if (empty($home_form) || empty($away_form)) {
            return ['error' => 'Insufficient form data'];
        }
        
        // Calculate form points (W=3, D=1, L=0)
        $home_form_points = array_sum(array_map(function($result) {
            return $result == 'W' ? 3 : ($result == 'D' ? 1 : 0);
        }, $home_form));
        
        $away_form_points = array_sum(array_map(function($result) {
            return $result == 'W' ? 3 : ($result == 'D' ? 1 : 0);
        }, $away_form));
        
        // Calculate form strength
        $home_form_strength = ($home_form_points / (count($home_form) * 3)) * 100;
        $away_form_strength = ($away_form_points / (count($away_form) * 3)) * 100;
        
        // Add home advantage
        $home_form_strength += 8;
        
        $total = $home_form_strength + $away_form_strength;
        $home_prob = ($home_form_strength / $total) * 100;
        $away_prob = ($away_form_strength / $total) * 100;
        $draw_prob = 100 - $home_prob - $away_prob;
        
        $prediction = $this->getPredictionFromProbabilities($home_prob, $draw_prob, $away_prob);
        
        return [
            'model' => 'Recent Form Analysis',
            'prediction' => $prediction,
            'home_win_prob' => round($home_prob, 2),
            'draw_prob' => round($draw_prob, 2),
            'away_win_prob' => round($away_prob, 2),
            'home_form' => implode('', $home_form),
            'away_form' => implode('', $away_form),
            'confidence' => round(max($home_prob, $draw_prob, $away_prob), 2)
        ];
    }
    
    /**
     * Generate Recommended Bet
     * Combines all models and suggests best bet
     */
    public function getRecommendedBet($user_odds) {
        $poisson = $this->poissonPrediction();
        $win_pct = $this->winPercentagePrediction();
        $xg = $this->xgPrediction();
        $form = $this->formPrediction();
        
        // Average probabilities from all models
        $avg_home = ($poisson['home_win_prob'] + $win_pct['home_win_prob'] + $xg['home_win_prob'] + $form['home_win_prob']) / 4;
        $avg_draw = ($poisson['draw_prob'] + $win_pct['draw_prob'] + $xg['draw_prob'] + $form['draw_prob']) / 4;
        $avg_away = ($poisson['away_win_prob'] + $win_pct['away_win_prob'] + $xg['away_win_prob'] + $form['away_win_prob']) / 4;
        
        // Calculate implied probabilities from odds
        $implied_home = (1 / $user_odds['home']) * 100;
        $implied_draw = (1 / $user_odds['draw']) * 100;
        $implied_away = (1 / $user_odds['away']) * 100;
        
        // Find value bets (where our probability > implied probability)
        $recommendations = [];
        
        if ($avg_home > $implied_home) {
            $value = (($avg_home / $implied_home) - 1) * 100;
            $recommendations[] = ['bet' => 'Home Win', 'value' => $value, 'odds' => $user_odds['home'], 'prob' => $avg_home];
        }
        
        if ($avg_draw > $implied_draw) {
            $value = (($avg_draw / $implied_draw) - 1) * 100;
            $recommendations[] = ['bet' => 'Draw', 'value' => $value, 'odds' => $user_odds['draw'], 'prob' => $avg_draw];
        }
        
        if ($avg_away > $implied_away) {
            $value = (($avg_away / $implied_away) - 1) * 100;
            $recommendations[] = ['bet' => 'Away Win', 'value' => $value, 'odds' => $user_odds['away'], 'prob' => $avg_away];
        }
        
        // Sort by value
        usort($recommendations, function($a, $b) {
            return $b['value'] <=> $a['value'];
        });
        
        $confidence = round(max($avg_home, $avg_draw, $avg_away), 2);
        
        return [
            'recommendation' => $recommendations[0] ?? null,
            'all_options' => $recommendations,
            'avg_probabilities' => [
                'home' => round($avg_home, 2),
                'draw' => round($avg_draw, 2),
                'away' => round($avg_away, 2)
            ],
            'confidence' => $confidence,
            'models' => [
                'poisson' => $poisson,
                'win_percentage' => $win_pct,
                'xg' => $xg,
                'form' => $form
            ]
        ];
    }
    
    // ========== Helper Functions ==========
    
    private function poissonProbability($k, $lambda) {
        return (pow($lambda, $k) * exp(-$lambda)) / $this->factorial($k);
    }
    
    private function factorial($n) {
        if ($n <= 1) return 1;
        return $n * $this->factorial($n - 1);
    }
    
    private function getPredictionFromProbabilities($home, $draw, $away) {
        $max = max($home, $draw, $away);
        if ($max == $home) return 'Home Win';
        if ($max == $draw) return 'Draw';
        return 'Away Win';
    }
    
    private function xgToWinProbability($team_xg, $opponent_xg) {
        // Simplified xG to win probability conversion
        $diff = $team_xg - $opponent_xg;
        return 50 + ($diff * 15); // Rough approximation
    }
    
    private function getTeamStats($team_id) {
        $query = "SELECT * FROM team_stats WHERE team_id = ? AND season = ? LIMIT 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("is", $team_id, $this->season);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }
    
    private function getH2HRecord() {
        $query = "SELECT * FROM h2h_records WHERE (team1_id = ? AND team2_id = ?) OR (team1_id = ? AND team2_id = ?)";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("iiii", $this->home_team_id, $this->away_team_id, $this->away_team_id, $this->home_team_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $record = $result->fetch_assoc();
        
        if ($record) {
            $record['total_h2h'] = $record['team1_wins'] + $record['team2_wins'] + $record['draws'];
            return $record;
        }
        
        return ['team1_wins' => 0, 'team2_wins' => 0, 'draws' => 0, 'total_h2h' => 0];
    }
    
    private function getRecentForm($team_id, $limit = 10) {
        $query = "SELECT result FROM matches WHERE (home_team_id = ? OR away_team_id = ?) AND result IS NOT NULL ORDER BY match_date DESC LIMIT ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("iii", $team_id, $team_id, $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $form = [];
        while ($row = $result->fetch_assoc()) {
            $form[] = $row['result'];
        }
        
        return array_reverse($form);
    }
}
?>
