/**
 * Odds Predictor - Main Application JavaScript
 */

// Load teams when league is selected
function loadTeams() {
    const leagueId = document.getElementById('league').value;
    
    if (!leagueId) {
        document.getElementById('home_team').innerHTML = '<option value="">-- Select Team --</option>';
        document.getElementById('away_team').innerHTML = '<option value="">-- Select Team --</option>';
        return;
    }
    
    fetch('/odds-predictor/public/api/get-teams.php?league_id=' + leagueId)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                let homeOptions = '<option value="">-- Select Home Team --</option>';
                let awayOptions = '<option value="">-- Select Away Team --</option>';
                
                data.teams.forEach(team => {
                    homeOptions += `<option value="${team.id}">${team.name}</option>`;
                    awayOptions += `<option value="${team.id}">${team.name}</option>`;
                });
                
                document.getElementById('home_team').innerHTML = homeOptions;
                document.getElementById('away_team').innerHTML = awayOptions;
            }
        })
        .catch(error => console.error('Error loading teams:', error));
}

// Handle form submission
document.getElementById('predictionForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    
    // Validate odds
    const odds_home = parseFloat(formData.get('odds_home'));
    const odds_draw = parseFloat(formData.get('odds_draw'));
    const odds_away = parseFloat(formData.get('odds_away'));
    
    if (odds_home < 1.0 || odds_draw < 1.0 || odds_away < 1.0) {
        showError('Odds must be 1.0 or higher');
        return;
    }
    
    // Show loading state
    const resultsDiv = document.getElementById('results');
    resultsDiv.innerHTML = '<div class="loading"><p>Analyzing data and generating predictions...</p></div>';
    
    fetch('/odds-predictor/public/predict.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            displayResults(data);
        } else {
            showError(data.error || 'An error occurred');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showError('Failed to get prediction. Please try again.');
    });
});

// Display prediction results
function displayResults(data) {
    const resultsDiv = document.getElementById('results');
    
    const recommendation = data.recommendation;
    const models = data.models;
    const avgProbs = data.average_probabilities;
    const confidence = data.confidence_level;
    
    let html = `
        <div class="results-section">
            <h2>Prediction Results</h2>
            <div class="match-info">
                <h3>${data.match.home_team} vs ${data.match.away_team}</h3>
                <p>Your Odds: ${data.user_odds.home} - ${data.user_odds.draw} - ${data.user_odds.away}</p>
            </div>
            
            <div class="recommendation-box ${recommendation ? 'has-recommendation' : 'no-recommendation'}">
                <h3>🎯 Recommended Bet</h3>
                ${recommendation ? `
                    <div class="recommendation-content">
                        <p class="recommendation-bet">${recommendation.bet}</p>
                        <p class="recommendation-odds">Odds: ${recommendation.odds.toFixed(2)}</p>
                        <p class="recommendation-value">Value: ${recommendation.value.toFixed(1)}%</p>
                        <p class="recommendation-prob">Probability: ${recommendation.prob.toFixed(1)}%</p>
                        <p class="confidence-level">Confidence: ${confidence.toFixed(1)}%</p>
                    </div>
                ` : `
                    <p>No clear value bet found at current odds</p>
                `}
            </div>
            
            <div class="all-options">
                <h3>All Betting Options</h3>
                <div class="options-grid">
                    ${data.all_options.map(option => `
                        <div class="option-card">
                            <h4>${option.bet}</h4>
                            <p>Odds: ${option.odds.toFixed(2)}</p>
                            <p>Probability: ${option.prob.toFixed(1)}%</p>
                            <p>Value: ${option.value.toFixed(1)}%</p>
                        </div>
                    `).join('')}
                </div>
            </div>
            
            <div class="average-probabilities">
                <h3>Average Probabilities (All Models)</h3>
                <div class="probability-bars">
                    <div class="prob-bar">
                        <label>Home Win</label>
                        <div class="bar-container">
                            <div class="bar" style="width: ${avgProbs.home}%"></div>
                            <span>${avgProbs.home.toFixed(1)}%</span>
                        </div>
                    </div>
                    <div class="prob-bar">
                        <label>Draw</label>
                        <div class="bar-container">
                            <div class="bar" style="width: ${avgProbs.draw}%"></div>
                            <span>${avgProbs.draw.toFixed(1)}%</span>
                        </div>
                    </div>
                    <div class="prob-bar">
                        <label>Away Win</label>
                        <div class="bar-container">
                            <div class="bar" style="width: ${avgProbs.away}%"></div>
                            <span>${avgProbs.away.toFixed(1)}%</span>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="models-section">
                <h3>Individual Model Predictions</h3>
                <div class="models-grid">
                    ${Object.entries(models).map(([key, model]) => `
                        <div class="model-card">
                            <h4>${model.model}</h4>
                            <p><strong>Prediction:</strong> ${model.prediction}</p>
                            <p>Home: ${model.home_win_prob.toFixed(1)}%</p>
                            <p>Draw: ${model.draw_prob.toFixed(1)}%</p>
                            <p>Away: ${model.away_win_prob.toFixed(1)}%</p>
                            <p class="confidence">Confidence: ${model.confidence.toFixed(1)}%</p>
                        </div>
                    `).join('')}
                </div>
            </div>
            
            <div class="disclaimer-box">
                <p><strong>⚠️ Disclaimer:</strong> ${data.disclaimer}</p>
                <p>This tool uses historical data and statistical models to generate predictions. 
                   Past performance does not guarantee future results. Always gamble responsibly.</p>
            </div>
        </div>
    `;
    
    resultsDiv.innerHTML = html;
    resultsDiv.scrollIntoView({ behavior: 'smooth', block: 'start' });
}

// Show error message
function showError(message) {
    const resultsDiv = document.getElementById('results');
    resultsDiv.innerHTML = `<div class="error-box"><p>❌ ${message}</p></div>`;
    resultsDiv.scrollIntoView({ behavior: 'smooth', block: 'start' });
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    console.log('Odds Predictor loaded');
});
