<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/PredictionEngine.php';

header('Content-Type: application/json');

// Verify age
if (!isAgeVerified()) {
    http_response_code(403);
    echo json_encode(['error' => 'Age verification required']);
    exit;
}

// Verify CSRF token
if (!isset($_POST['csrf_token']) || !validateCSRFToken($_POST['csrf_token'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid CSRF token']);
    exit;
}

// Validate input
$home_team_id = isset($_POST['home_team']) ? (int)$_POST['home_team'] : 0;
$away_team_id = isset($_POST['away_team']) ? (int)$_POST['away_team'] : 0;
$league_id = isset($_POST['league']) ? (int)$_POST['league'] : 0;
$odds_home = isset($_POST['odds_home']) ? (float)$_POST['odds_home'] : 0;
$odds_draw = isset($_POST['odds_draw']) ? (float)$_POST['odds_draw'] : 0;
$odds_away = isset($_POST['odds_away']) ? (float)$_POST['odds_away'] : 0;

if (!$home_team_id || !$away_team_id || !$league_id || !$odds_home || !$odds_draw || !$odds_away) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing required fields']);
    exit;
}

try {
    $conn = getDBConnection();
    
    // Get current season (default to 2024/25)
    $season = '2024/25';
    
    // Create prediction engine
    $engine = new PredictionEngine($conn, $home_team_id, $away_team_id, $season);
    
    // Get recommendation
    $user_odds = [
        'home' => $odds_home,
        'draw' => $odds_draw,
        'away' => $odds_away
    ];
    
    $recommendation = $engine->getRecommendedBet($user_odds);
    
    // Get team names
    $home_query = "SELECT name FROM teams WHERE id = ?";
    $stmt = $conn->prepare($home_query);
    $stmt->bind_param("i", $home_team_id);
    $stmt->execute();
    $home_result = $stmt->get_result()->fetch_assoc();
    
    $away_query = "SELECT name FROM teams WHERE id = ?";
    $stmt = $conn->prepare($away_query);
    $stmt->bind_param("i", $away_team_id);
    $stmt->execute();
    $away_result = $stmt->get_result()->fetch_assoc();
    
    // Log prediction
    $session_id = session_id();
    $log_query = "INSERT INTO predictions (user_session_id, home_team_id, away_team_id, league_id, user_odds_home, user_odds_draw, user_odds_away, recommended_bet, confidence_level) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $conn->prepare($log_query);
    $recommended_bet = $recommendation['recommendation']['bet'] ?? 'No Clear Recommendation';
    $confidence = $recommendation['confidence'] ?? 0;
    $stmt->bind_param("siiiddds", $session_id, $home_team_id, $away_team_id, $league_id, $odds_home, $odds_draw, $odds_away, $recommended_bet, $confidence);
    $stmt->execute();
    
    logActivity('prediction_generated', "Teams: {$home_result['name']} vs {$away_result['name']}");
    
    // Prepare response
    $response = [
        'success' => true,
        'match' => [
            'home_team' => $home_result['name'],
            'away_team' => $away_result['name']
        ],
        'user_odds' => $user_odds,
        'recommendation' => $recommendation['recommendation'],
        'all_options' => $recommendation['all_options'],
        'average_probabilities' => $recommendation['avg_probabilities'],
        'confidence_level' => $recommendation['confidence'],
        'models' => $recommendation['models'],
        'disclaimer' => 'This is for entertainment purposes only. Not financial advice. Gamble responsibly.'
    ];
    
    echo json_encode($response);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Prediction failed: ' . $e->getMessage()]);
    logActivity('prediction_error', $e->getMessage());
}
?>
